/*
	File:		ColorPenState.c

	Contains:	Utility routines to save and restore the graphics state of a grafport.

	Version:	Appearance 1.0 SDK

	Copyright:	 1997 by Apple Computer, Inc., all rights reserved.

	File Ownership:

		DRI:				Edward Voas

		Other Contact:		7 of 9, Borg Collective

		Technology:			OS Technologies Group

	Writers:

		(edv)	Ed Voas

	Change History (most recent first):

		 <1>	 9/11/97	edv		First checked in.
*/

#include "ColorPenState.h"
#include "ColorUtils.h"
#include "Assertions.h"

//-----------------------------------------------------------------------------------------
//	 GetColorAndPenState
//
//	Gets the current drawing environment and stores the data in state. We copy pen and back
//	pix pats only if they are not type 0 (plain ol expanded black and white patterns).
//-----------------------------------------------------------------------------------------

void
GetColorAndPenState( ColorPenState* state )
{
	GrafPtr			curPort;
	
	GetPort( &curPort );
	
	state->pnPixPat = nil;
	state->bkPixPat = nil;

	state->colorPort = IsColorGrafPort( curPort );

	state->bkPat = curPort->bkPat;
	state->bkColor = curPort->bkColor;
	state->fgColor = curPort->fgColor;

	if ( state->colorPort )
	{
		GetForeColor( &state->foreColor );
		GetBackColor( &state->backColor );
		
			// If the pen pattern is not an old style pattern,
			// copy the handle. If it is an old style pattern,
			// GetPenState below will save the right thing.
			
		if ( (**((CGrafPtr)curPort)->pnPixPat).patType != 0 )
		{
			state->pnPixPat = ((CGrafPtr)curPort)->pnPixPat;
		}
		
			// If the pen pattern is not an old style pattern,
			// copy the handle, else get the old pattern into
			// bkPat for restoring that way.
			
		if ( (**((CGrafPtr)curPort)->bkPixPat).patType != 0 )
		{
			state->bkPixPat = ((CGrafPtr)curPort)->bkPixPat;
		}
		else
			state->bkPat = *(PatPtr)(*(**((CGrafPtr)curPort)->bkPixPat).patData);
	}
		
	GetPenState( &state->pen );
	state->textMode = curPort->txMode;
}

//-----------------------------------------------------------------------------------------
//	 SetColorAndPenState
//
//	Sets the current drawing environment based on the data in state.
//-----------------------------------------------------------------------------------------

void
SetColorAndPenState( ColorPenState* state )
{
	GrafPtr		curPort;
	
	GetPort( &curPort );

	SetPenState( &state->pen );

	if ( IsColorGrafPort( curPort ) && state->colorPort )
	{
		RGBForeColor( &state->foreColor );
		RGBBackColor( &state->backColor );

		if ( state->pnPixPat )
			PenPixPat( state->pnPixPat );

		if ( state->bkPixPat )
			BackPixPat( state->bkPixPat );
		else
			BackPat( &state->bkPat );
	}
	else
	{
		BackPat( &state->bkPat );
		ForeColor( state->fgColor );
		BackColor( state->bkColor );
	}

	TextMode( state->textMode );
}

//-----------------------------------------------------------------------------------------
//	 NormalizeColorAndPen
//
//	Sets up our environment to standard drawing fare.
//-----------------------------------------------------------------------------------------

void
NormalizeColorAndPen()
{
	RGBColor		black, white;
	Pattern			whitePat = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	GrafPtr			curPort;
	
	GetPort( &curPort );
	
	black.red = black.green = black.blue = 0x0000;
	white.red = white.green = white.blue = 0xFFFF;
	
	if ( IsColorGrafPort( curPort ) )
	{
		RGBForeColor( &black );
		RGBBackColor( &white );
	}
	PenNormal();
	BackPat( &whitePat );
	TextMode( srcOr );
}
